//GEMINI-pro DATASET v1.0 Category: Building a HTTP Client ; Style: future-proof
#include <curl/curl.h>
#include <stdio.h>
#include <stdlib.h>

#define CURL_TIMEOUT 10L

// Define the structure to store the response data
typedef struct {
  char *data;
  size_t size;
} ResponseData;

// Callback function to handle the response data
static size_t write_callback(void *buffer, size_t size, size_t nitems, void *userdata) {
  // Get the response data structure
  ResponseData *response_data = (ResponseData *)userdata;

  // Calculate the total size of the response data
  size_t total_size = size * nitems;

  // Reallocate the response data buffer if necessary
  if (total_size > response_data->size) {
    response_data->data = realloc(response_data->data, total_size + 1);
    if (response_data->data == NULL) {
      return 0;
    }
  }

  // Copy the response data into the buffer
  memcpy(response_data->data + response_data->size, buffer, total_size);

  // Update the size of the response data
  response_data->size += total_size;

  // Terminate the response data with a null character
  response_data->data[response_data->size] = '\0';

  return total_size;
}

int main(int argc, char **argv) {
  // Initialize the CURL library
  if (curl_global_init(CURL_GLOBAL_ALL) != CURLE_OK) {
    fprintf(stderr, "Error: Failed to initialize CURL library\n");
    return EXIT_FAILURE;
  }

  // Create a CURL handle
  CURL *curl = curl_easy_init();
  if (curl == NULL) {
    fprintf(stderr, "Error: Failed to create CURL handle\n");
    curl_global_cleanup();
    return EXIT_FAILURE;
  }

  // Set the URL to fetch
  if (argc < 2) {
    fprintf(stderr, "Usage: %s <URL>\n", argv[0]);
    curl_easy_cleanup(curl);
    curl_global_cleanup();
    return EXIT_FAILURE;
  }
  char *url = argv[1];

  // Set the URL to fetch
  curl_easy_setopt(curl, CURLOPT_URL, url);

  // Set the timeout for the request
  curl_easy_setopt(curl, CURLOPT_TIMEOUT, CURL_TIMEOUT);

  // Set the callback function to handle the response data
  ResponseData response_data = {NULL, 0};
  curl_easy_setopt(curl, CURLOPT_WRITEFUNCTION, write_callback);
  curl_easy_setopt(curl, CURLOPT_WRITEDATA, &response_data);

  // Perform the HTTP request
  CURLcode result = curl_easy_perform(curl);
  if (result != CURLE_OK) {
    fprintf(stderr, "Error: Failed to perform HTTP request: %s\n", curl_easy_strerror(result));
    curl_easy_cleanup(curl);
    curl_global_cleanup();
    return EXIT_FAILURE;
  }

  // Get the HTTP response code
  long response_code;
  curl_easy_getinfo(curl, CURLINFO_RESPONSE_CODE, &response_code);

  // Print the HTTP response code and the response data
  printf("HTTP response code: %ld\n", response_code);
  printf("Response data:\n%s\n", response_data.data);

  // Cleanup the CURL handle and the CURL library
  curl_easy_cleanup(curl);
  curl_global_cleanup();

  // Free the response data
  free(response_data.data);

  return EXIT_SUCCESS;
}