//GPT-4o-mini DATASET v1.0 Category: Classical Circuit Simulator ; Style: shape shifting
#include <stdio.h>
#include <stdlib.h>
#include <stdbool.h>
#include <math.h>

#define MAX_COMPONENTS 50

typedef enum {
    RESISTOR,
    CAPACITOR,
    INDUCTOR,
    VOLTAGE_SOURCE,
    CURRENT_SOURCE
} ComponentType;

typedef struct {
    ComponentType type;
    char name[20];
    double value; // in ohms for resistors, farads for capacitors, henrys for inductors, volts for voltage sources, amperes for current sources
} Component;

typedef struct {
    Component components[MAX_COMPONENTS];
    int component_count; 
} Circuit;

void initCircuit(Circuit* circuit) {
    circuit->component_count = 0;
}

bool addComponent(Circuit* circuit, ComponentType type, const char* name, double value) {
    if (circuit->component_count >= MAX_COMPONENTS) {
        printf("Error: Maximum component limit reached!\n");
        return false;
    }
    
    Component new_component;
    new_component.type = type;
    snprintf(new_component.name, sizeof(new_component.name), "%s", name);
    new_component.value = value;
    
    circuit->components[circuit->component_count++] = new_component;
    printf("Added %s: %.2f of type %d\n", name, value, type);
    
    return true;
}

void simulateCircuit(Circuit* circuit) {
    printf("\nSimulating Circuit...\n");
    double totalResistance = 0.0;
    double totalCapacitance = 0.0;
    double totalInductance = 0.0;
    double voltage = 0.0;
    
    for (int i = 0; i < circuit->component_count; i++) {
        Component* component = &circuit->components[i];
        switch (component->type) {
            case RESISTOR:
                totalResistance += component->value;
                break;
            case CAPACITOR:
                totalCapacitance += component->value;
                break;
            case INDUCTOR:
                totalInductance += component->value;
                break;
            case VOLTAGE_SOURCE:
                voltage += component->value;
                break;
            case CURRENT_SOURCE:
                // Skipping current sources in basic analysis
                break;
        }
    }
    
    printf("Total Resistance: %.2f Ohms\n", totalResistance);
    printf("Total Capacitance: %.2f F\n", totalCapacitance);
    printf("Total Inductance: %.2f H\n", totalInductance);
    printf("Total Voltage: %.2f V\n", voltage);
}

void displayCircuit(Circuit* circuit) {
    printf("\nCurrent Circuit Configuration:\n");
    for (int i = 0; i < circuit->component_count; i++) {
        Component* component = &circuit->components[i];
        printf("Component %d: %s - %.2f of type %d\n", 
               i + 1, component->name, component->value, component->type);
    }
}

int main() {
    Circuit circuit;
    initCircuit(&circuit);

    // Adding components in a shape-shifting fashion
    addComponent(&circuit, RESISTOR, "R1", 100.0);
    addComponent(&circuit, CAPACITOR, "C1", 1e-6);
    addComponent(&circuit, VOLTAGE_SOURCE, "V1", 9.0);
    addComponent(&circuit, INDUCTOR, "L1", 0.01);
    addComponent(&circuit, CURRENT_SOURCE, "I1", 0.5);
    addComponent(&circuit, RESISTOR, "R2", 220.0);
    addComponent(&circuit, CAPACITOR, "C2", 2e-6);

    displayCircuit(&circuit);
    simulateCircuit(&circuit);

    return 0;
}