//GPT-4o-mini DATASET v1.0 Category: Cryptographic hash function ; Style: thoughtful
#include <stdio.h>
#include <string.h>
#include <stdint.h>

#define BLOCK_SIZE 64 // SHA-256 operates on 512-bit blocks (64 bytes)
#define HASH_SIZE 32  // Output is 32 bytes

// Function Prototypes
void sha256(const uint8_t *data, size_t len, uint8_t *hash);
void initialize_hash(uint32_t *hash);
void transform(uint32_t *hash, const uint8_t *data);
void compute_final_hash(uint32_t *hash, uint8_t *out);

// SHA-256 Constants
static const uint32_t k[64] = {
    0x428a2f98, 0x71374491, 0xb5c0fbcf, 0xe9b5dba5,
    0x3956c25b, 0x59f111f1, 0x923f82a4, 0xab1c5ed5,
    0xd807aa98, 0x12835b01, 0x243185be, 0x550c7dc3,
    0x72be5d74, 0x80deb1fe, 0x9bdc06a7, 0xc19bf174,
    0xe49b69c1, 0xefbe4786, 0x0fc19dc6, 0x240ca1cc,
    0x2de92c6f, 0x4a7484aa, 0x5cb0a9dc, 0x76f988da,
    0x983e5152, 0xa831c66d, 0xb00327c8, 0xbf597fc7,
    0xc6e00bf3, 0xd5a79147, 0x06ca6351, 0x14292967,
    0x27b70a85, 0x2e1b2138, 0x4d2c6dfc, 0x53380d13,
    0x650a7354, 0x766a0abb, 0x81c2c92e, 0x92722c85,
    0xa2bfe8a1, 0xa81a664b, 0xc24b8b70, 0xc76c51a3,
    0xd192e819, 0xd6990624, 0xf40e3585, 0x106aa070,
    0x19a4c116, 0x1e376c48, 0x2748774c, 0x34b0bcb5,
    0x391c0cb3, 0x4ed8aa11, 0x5b9cca4f, 0x682e6ff3,
    0x748f82ee, 0x78a5636f, 0x7c9f8e5b, 0x8cc70208,
    0x90befffa, 0xa4506ceb, 0xbef9a3f7, 0xc67178f2
};

// Initialize hash values
void initialize_hash(uint32_t *hash) {
    hash[0] = 0x6a09e667;
    hash[1] = 0xbb67ae85;
    hash[2] = 0x3c6ef372;
    hash[3] = 0xa54ff53a;
    hash[4] = 0x510e527f;
    hash[5] = 0x9b05688c;
    hash[6] = 0x1f83d9ab;
    hash[7] = 0x5be0cd19;
}

// Right rotate helper function
uint32_t right_rotate(uint32_t value, uint32_t amount) {
    return (value >> amount) | (value << (32 - amount));
}

// SHA-256 transformation function
void transform(uint32_t *hash, const uint8_t *data) {
    uint32_t w[64] = {0};
    for (int i = 0; i < 16; ++i) {
        w[i] = (data[i * 4] << 24) | (data[i * 4 + 1] << 16) |
               (data[i * 4 + 2] << 8) | (data[i * 4 + 3]);
    }

    for (int i = 16; i < 64; ++i) {
        uint32_t s0 = right_rotate(w[i - 15], 7) ^ right_rotate(w[i - 15], 18) ^ (w[i - 15] >> 3);
        uint32_t s1 = right_rotate(w[i - 2], 17) ^ right_rotate(w[i - 2], 19) ^ (w[i - 2] >> 10);
        w[i] = w[i - 16] + s0 + w[i - 7] + s1;
    }

    uint32_t a = hash[0], b = hash[1], c = hash[2], d = hash[3];
    uint32_t e = hash[4], f = hash[5], g = hash[6], h = hash[7];

    for (int i = 0; i < 64; ++i) {
        uint32_t S1 = right_rotate(e, 6) ^ right_rotate(e, 11) ^ right_rotate(e, 25);
        uint32_t ch = (e & f) ^ (~e & g);
        uint32_t temp1 = h + S1 + ch + k[i] + w[i];
        uint32_t S0 = right_rotate(a, 2) ^ right_rotate(a, 13) ^ right_rotate(a, 22);
        uint32_t maj = (a & b) ^ (a & c) ^ (b & c);
        uint32_t temp2 = S0 + maj;

        h = g;
        g = f;
        f = e;
        e = d + temp1;
        d = c;
        c = b;
        b = a;
        a = temp1 + temp2;
    }

    hash[0] += a;
    hash[1] += b;
    hash[2] += c;
    hash[3] += d;
    hash[4] += e;
    hash[5] += f;
    hash[6] += g;
    hash[7] += h;
}

// SHA-256 hash calculation
void sha256(const uint8_t *data, size_t len, uint8_t *hash) {
    uint32_t h[8];
    initialize_hash(h);
    
    uint8_t padded_data[BLOCK_SIZE] = {0};
    size_t offset = 0;
    
    while (len >= BLOCK_SIZE) {
        transform(h, data + offset);
        offset += BLOCK_SIZE;
        len -= BLOCK_SIZE;
    }

    memcpy(padded_data, data + offset, len);
    padded_data[len] = 0x80; // Add a single 1 bit

    if (len >= 56) {
        memset(padded_data + len + 1, 0, BLOCK_SIZE - len - 1);
        transform(h, padded_data);
        memset(padded_data, 0, BLOCK_SIZE);
    } else {
        memset(padded_data + len + 1, 0, BLOCK_SIZE - len - 9);
    }
    
    uint64_t total_bits = (uint64_t)len * 8; // Original length in bits
    for (int i = 0; i < 8; i++)
        padded_data[BLOCK_SIZE - 1 - i] = (total_bits >> (i * 8)) & 0xff;

    transform(h, padded_data);

    for (int i = 0; i < 8; i++) {
        hash[i * 4] = (h[i] >> 24) & 0xff;
        hash[i * 4 + 1] = (h[i] >> 16) & 0xff;
        hash[i * 4 + 2] = (h[i] >> 8) & 0xff;
        hash[i * 4 + 3] = h[i] & 0xff;
    }
}

// Main function for testing the hash function
int main() {
    const char *message = "Hello, this is a test for SHA-256!";
    uint8_t hash[HASH_SIZE];

    sha256((uint8_t *)message, strlen(message), hash);

    printf("SHA-256 Hash of \"%s\":\n", message);
    for (int i = 0; i < HASH_SIZE; i++) {
        printf("%02x", hash[i]);
    }
    printf("\n");

    return 0;
}