//GEMINI-pro DATASET v1.0 Category: Image to ASCII art converter ; Style: high level of detail
#include <stdio.h>
#include <stdlib.h>
#include <string.h>

// Define the maximum width and height of the image
#define MAX_WIDTH 1024
#define MAX_HEIGHT 1024

// Define the number of ASCII characters to use
#define NUM_ASCII_CHARS 256

// Define the ASCII characters to use
const char *ascii_chars = "@#%^&*+-=:.  ";

// Convert an image to ASCII art
char *image_to_ascii(const char *filename) {
  // Open the image file
  FILE *fp = fopen(filename, "rb");
  if (fp == NULL) {
    fprintf(stderr, "Error opening file %s\n", filename);
    return NULL;
  }

  // Read the image header
  unsigned char header[18];
  fread(header, sizeof(unsigned char), 18, fp);

  // Check if the file is a valid BMP file
  if (header[0] != 'B' || header[1] != 'M') {
    fprintf(stderr, "Error: file is not a valid BMP file\n");
    fclose(fp);
    return NULL;
  }

  // Get the image width and height
  int width = *(int *)&header[18];
  int height = *(int *)&header[22];

  // Allocate memory for the image data
  unsigned char *data = malloc(width * height * 3);
  if (data == NULL) {
    fprintf(stderr, "Error allocating memory for image data\n");
    fclose(fp);
    return NULL;
  }

  // Read the image data
  fread(data, sizeof(unsigned char), width * height * 3, fp);

  // Close the image file
  fclose(fp);

  // Convert the image data to ASCII art
  char *ascii_art = malloc(width * height + 1);
  if (ascii_art == NULL) {
    fprintf(stderr, "Error allocating memory for ASCII art\n");
    free(data);
    return NULL;
  }

  for (int i = 0; i < width * height; i++) {
    // Get the RGB values of the pixel
    unsigned char r = data[i * 3 + 2];
    unsigned char g = data[i * 3 + 1];
    unsigned char b = data[i * 3 + 0];

    // Calculate the average RGB value of the pixel
    unsigned char avg = (r + g + b) / 3;

    // Map the average RGB value to an ASCII character
    ascii_art[i] = ascii_chars[avg / (NUM_ASCII_CHARS - 1)];
  }

  // Add a newline character to the end of the ASCII art
  ascii_art[width * height] = '\0';

  // Free the memory allocated for the image data
  free(data);

  // Return the ASCII art
  return ascii_art;
}

// Save the ASCII art to a file
void save_ascii_art(const char *filename, char *ascii_art) {
  // Open the ASCII art file
  FILE *fp = fopen(filename, "w");
  if (fp == NULL) {
    fprintf(stderr, "Error opening file %s\n", filename);
    free(ascii_art);
    return;
  }

  // Write the ASCII art to the file
  fprintf(fp, "%s", ascii_art);

  // Close the ASCII art file
  fclose(fp);
}

// Main function
int main(int argc, char *argv[]) {
  // Check if the user specified an input file
  if (argc < 2) {
    fprintf(stderr, "Usage: %s <input_file> [output_file]\n", argv[0]);
    return 1;
  }

  // Convert the image to ASCII art
  char *ascii_art = image_to_ascii(argv[1]);
  if (ascii_art == NULL) {
    return 1;
  }

  // Save the ASCII art to a file
  if (argc >= 3) {
    save_ascii_art(argv[2], ascii_art);
  } else {
    // Print the ASCII art to the console
    printf("%s", ascii_art);
  }

  // Free the memory allocated for the ASCII art
  free(ascii_art);

  return 0;
}